//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2023 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------


#ifndef RenderCore_hpp
#define RenderCore_hpp

#include <Metal/Metal.hpp>
#include <QuartzCore/QuartzCore.hpp>
#define IR_RUNTIME_METALCPP
#include <metal_irconverter_runtime/metal_irconverter_runtime.h>

#include "BumpAllocator.hpp"
#include "MeshUtils.hpp"
#include "ShaderPipelineBuilder.hpp"
#include <string>

constexpr size_t kMaxFramesInFlight = 3;

class RenderCore
{
public:
    enum RenderMode
    {
        RM_NoGS_NoTS = 0,
        RM_GS_NoTS   = 1,
        RM_GS_TS     = 2,
        RM_GS_TS_NonIndexed = 3,
        RM_GS_NoTS_NonIndexed = 4
    };

    RenderCore(MTL::Device* pDevice, const std::string& shaderSearchPath);
    ~RenderCore();

    // Set up the scene.
    void buildRenderPipelines(const std::string& shaderSearchPath);
    void buildComputePipelines(const std::string& shaderSearchPath);
    void buildDepthStencilStates();
    void buildTextures();
    void buildSamplers();
    void buildBuffers();

    // Call every frame to produce the animations.
    void generateMandelbrotTexture(MTL::CommandBuffer* pCommandBuffer);
    void updateWorld();
    void draw(MTL::RenderPassDescriptor* pRenderPass, CA::MetalDrawable* pDrawable);
    void drawInstancedCubes(MTL::RenderCommandEncoder* pEnc);
    void drawGrass(MTL::RenderCommandEncoder* pEnc);

    // Respond to configuration changes.
    void resizeDrawable(float width, float height);
    void setRenderMode(RenderCore::RenderMode renderMode) { _renderMode = renderMode; }
    void setTriangleFillMode(MTL::TriangleFillMode fillMode) { _triangleFillMode = fillMode; }
    void setAnimationFactor(float factor) { _animationFactor = factor; }
    void setTessellationInnerValue(float value) { _tessellationInnerValue = value; }

private:
    MTL::Device* _pDevice;
    MTL::CommandQueue* _pCommandQueue;
    dispatch_semaphore_t _semaphore;

    // Pipeline states:
    MTL::RenderPipelineState* _pInstancingPipeline;
    MTL::ComputePipelineState* _pComputeMandelbrotPipeline;
    MTL::RenderPipelineState* _pSimpleFlatGrassPipeline;
    RenderPipelineStateWithConfig<IRRuntimeGeometryPipelineConfig> _geometryPipeline;
    RenderPipelineStateWithConfig<IRRuntimeTessellationPipelineConfig> _tessellationPipeline;
    MTL::DepthStencilState* _pDepthStencilState;
    MTL::RenderPipelineState* _pDebugPipeline;

    // Assets:
    MTL::Texture* _pTexture;
    MTL::SamplerState* _pSampler;
    IndexedMesh _cubeMesh;
    IndexedMesh _pretessellatedGrassMesh;
    NonIndexedMesh _presessellatedNonIndexedGrassMesh;
    IndexedMesh _tessellationPatchGrassMesh;
    NonIndexedMesh _tessellationNonIndexedPatchGrassMesh;
    IndexedMesh _quadMesh;
    MTL::Heap* _pScratch;
    
    // Data needed for tessellation:
    MTL::Buffer* _pTessellatorTables;

    // Per-frame data:
    BumpAllocator* _bufferAllocator[kMaxFramesInFlight];

    // Instanced data:
    MTL::Buffer* _pInstanceDataBuffer[kMaxFramesInFlight];
    MTL::Buffer* _pCameraDataBuffer[kMaxFramesInFlight];

    // Animation data:
    float _angle;
    int _frame;
    uint _animationIndex;
    float _animationFactor;
    float _tessellationInnerValue;

    // Render config:
    float _aspectRatio;
    RenderMode _renderMode;
    MTL::TriangleFillMode _triangleFillMode;
};

#endif /* RenderCore_hpp */
